function [] = freeframe2aratome(tiling,varargin)
%This function converts freeframe mosaic image sessions to aratome format
%for stitching. Couple of things to expect. You will point the directory to
%the root of the sessions folders, e.g., /deconv/... or
%/exported-sessions/..., furthermore metadata needs to be in the same
%directory structure, although does not need to be in the same directory,
%e.g., files in /deconv/, metadata in /exported-sessions/  pointing to the
%same folder is fine.
%Synatax:   [] = freeframe2aratome('overlap','10','date','1976-11-01');
%Input:     'tiling' = the tiling width and length, e.g., [3,1], or 3 rows
%               by 1 column.  No defults must have.
%           'date' = data, Default = today's date
%           'overlap' = The % overlap of the images to be stitched, e.g.,10
%               which is 10%, Default = 10; Must be not be string
%           'dir' & 'meta' = the directories for images and metadata
%               respectively.  If left empty, function will prompt you.
%           'keepsection' = keep original section numbers.  Meaning if you
%               had a set of files where you removed sections you could
%               either preserve the original section number, or you could
%               let the program rename the sections starting from 0.
%               Default = 1; note that currently keepsection will crash the
%               aratome pipeline unless you start section from 0.
%           'metaonly' = output only the metadata files.  Default = 0.
%               Don't produce metadata only.
%Output:    no output at this moment

[overlap,sessiondate,dir_tmp,meta_dir,ksect,metaonly] = parse(varargin);

%where are the files?
if isempty(dir_tmp)
    dir_tmp = uigetdir2('','Directory where your images are');    %get the directory
end
if isempty(meta_dir)
    meta_dir = uigetdir2('','Directory where the metadata are located'); 
end
%get the root directory because we are going to save to it
sepidx = strfind(dir_tmp,filesep);
root_dir = dir_tmp(1:sepidx(end)-1);  %no filesep in the root dir end
dir_name = dir_tmp(sepidx(end)+1:end);  %grab the name of the current directory name
%make the converted save file folder
mkdir(root_dir,'AratomeStyle');
sav_dir = [root_dir,filesep,'AratomeStyle'];
mkdir(sav_dir,dir_name);
sav_dir = [sav_dir,filesep,dir_name];

%initiate the tiling index will call on this for saving files into tiles
tileidx = cell(tiling(1)*tiling(2),1);
%tilemeta = cell(tiling(1)*tiling(2),1);     %tiling for metadata
%initiate elapsed time counter
elptime = 111;
%initiate the metadata info
metacell = {};
%initiate blank var
blank_txt = [];
%contrast max - right now we are only dealing with 16bit gray.  If we need
%to deal with more bit rates or if this starts to matter, change it here.
contrastmax = '65535';

%now look at the root, if there are only files it will just pass it the
%root, if there are directories follow the above structure
sessions = dir(dir_tmp);
dir_idx = [sessions.isdir];   %grab all of the isdir numbers
dir_idx(1:2) = 0;               %the first two directories are not what we want
session_names = {sessions.name};   %grab the all of the names in the root
session_names = session_names(dir_idx);    %the directory names in root

%prepare sessions and metadata directories
img_dirtmp = repmat([dir_tmp filesep],size(session_names'));    %replicated the image directory
meta_dirtmp = repmat([meta_dir filesep],size(session_names'));    %replicated the meta directory
img_sessions = cellstr([img_dirtmp char(session_names')]);      %now each cell contains the full directory path
meta_sessions = cellstr([meta_dirtmp char(session_names')]);      %now each cell contains the full directory path

%prepare for resort files and save out
for i = 1:size(session_names,2)  %these are the sessions
    %now grab the image channels and sort through everything
    curr_dir = img_sessions{i};     %current image directory
    dir_info = dir(curr_dir); %grab the current dir
    curr_names = {dir_info.name};   %grab the all of the names
    idx = [dir_info.isdir];   %grab all of the isdir numbers
    idx(1:2) = 0;   %toss the first two
    chan_dir = curr_names(idx);    %the channel directories
    %metadata
    curr_metadir = meta_sessions{i};   %current meta directory
    %parse the sessions metadata
    fid = fopen([curr_metadir,filesep,'session_metadata.txt']);
    txt_tmp = textscan(fid,'%s%s%s%s%s%s%s','delimiter','\t');  %get data the file data structure is obviously stereotyped
    imgwidth = txt_tmp{1}{2};
    imgheight = txt_tmp{2}{2};
    %calculate the number of pixel overlap from entered percentage
    overlapstr = num2str(round((str2double(imgwidth)*(overlap/100)+str2double(imgheight)*(overlap/100))/2));
    fclose(fid);
    %now generate the save out tile structure
    mkdir(sav_dir,session_names{i})  %make the current save out folder
    cur_sav_dir = [sav_dir,filesep,session_names{i}];
    tilecount = 1;
    for m = 1:tiling(1)  %go rows
        for n = 1:tiling(2)  %then columns
            if i==1  %do this only once
                tileidx{tilecount} =  ['Tile-',num2str(m-1),'-',num2str(n-1)];
                %tilemeta{tilecount} =  ['Tile-',num2str(m-1),'-',num2str(n-1)];
            end
            mkdir(cur_sav_dir,tileidx{tilecount});
            tilecount = tilecount+1;
        end
    end
    %reorder the directory and get parallel aratome channel labels
    [ch_ord,chan_dir] = ord_dir(chan_dir);
    %now grab the files
    for j = 1:size(chan_dir,2)
        curr_imgdir = [img_sessions{i},filesep,chan_dir{j}];     %current image directory
        curr_metadir = [meta_sessions{i},filesep,chan_dir{j}];     %current meta directory
        curr_ch = ch_ord{j};        %current channel label
        imgdir_info = dir(curr_imgdir); %grab the current dir
        curr_imgnames = {imgdir_info.name};   %grab the all of the names
        imgidx = [imgdir_info.isdir];   %grab all of the isdir numbers
        imagefiles = curr_imgnames(~imgidx);   %grab the files first
        %We only want tifs
        imagefiles = findfiletype(imagefiles,'.tif');  %only tifs
        %now metadata files
        metadir_info = dir(curr_metadir); %grab the current dir
        curr_metanames = {metadir_info.name};   %grab the all of the names
        metaidx = [metadir_info.isdir];   %grab all of the isdir numbers
        metafiles = curr_metanames(~metaidx);   %grab the files first
        %We only want txt files
        metafiles = findfiletype(metafiles,'.txt');  %only txts
        %initiate z position
        zposition = 0;
        %number of channels in the set
        num_ch = size(ch_ord,2);
        %initiate cell array for metadata
        if j==1  %only want to do this once per session
            metacell = cell((size(imagefiles,2)/size(tileidx,1))*num_ch,size(tileidx,1));  %so one metadata file per tile 
        end
        sectold = 'ini';
        sectidx = -1;
        for k = 1:size(imagefiles,2)  %cycle through files
            %parse file metadata
            fid = fopen([curr_metadir,filesep,metafiles{k}]);  %this is assuming there is a one to one correspondence of metafiles to files
            txt_tmp = textscan(fid,'%s%s%s%s%s%s%s','delimiter','\t');  %get data the file data structure is obviously stereotyped
            xposition = txt_tmp{1}{6};
            yposition = txt_tmp{2}{6};
            fclose(fid);
            %now parse image file name
            filename = imagefiles{k}(1:end-4);  %take out the .tif
            fieldnum = str2double(filename(end-2:end))+1;
            positionidx = str2double(filename(end-2:end));  %the position index of this section
            sectnum = filename(end-7:end-5);
            if ~strcmp(sectold,sectnum)  %not the same new section
                sectidx = sectidx+1;
                sectold = sectnum;
            end
            if ~ksect  %don't keep section number
                sectnum = num2str(sectidx);
            end
            %copy over the file
            if ~metaonly  %if files are desired.
                copyfile([curr_imgdir,filesep,imagefiles{k}],[cur_sav_dir,filesep,tileidx{fieldnum},filesep,'img_000000000_',curr_ch,'_',sectnum,'.tif']);
            end
            %now prepare it's metadata
            [metastr] = curr_file_meta(j,str2double(sectnum),curr_ch,elptime,overlapstr,sessiondate,imgwidth,tileidx{fieldnum},yposition,imgheight,xposition,zposition,sectnum,positionidx);
            elptime = elptime+111;
            zposition = zposition+0.1035;
            %place metadata into metacell
            metacell{j+num_ch*sectidx,fieldnum} = metastr;
        end
    end
    %construct metadata & save out
    for o = 1:size(metacell,2)
        [metacap] = curr_cap_meta(sectidx+1,sessiondate,imgwidth,ch_ord,imgheight,tiling,contrastmax,o-1);
        %one tile at a time
        tile_tmp = metacell(:,o);
        metafile = vertcat({'{'},metacap,tile_tmp,{'}'});
        %first create a blank text file with correct name
        csvwrite([cur_sav_dir,filesep,tileidx{o},filesep,'metadata.txt'],blank_txt);
        fid = fopen([cur_sav_dir,filesep,tileidx{o},filesep,'metadata.txt'],'w');  %open file for writing
        for p = 1:size(metafile,1)
            fprintf(fid,[metafile{p},'\r\n']);
        end
        fclose(fid);
    end
    %one final metata set
    [metaroot] = root_meta(ch_ord);
    %first create a blank text file with correct name
    csvwrite([cur_sav_dir,filesep,'display_and_comments.txt'],blank_txt);
    fid = fopen([cur_sav_dir,filesep,'display_and_comments.txt'],'w');  %open file for writing
    fprintf(fid,metaroot);
    fclose(fid);
end

%--------------------------------------------------------------------------
%subfunction to generate metadata cap
function [metacap] = curr_cap_meta(sect,sess_date,imgwidth,chan_ord,imgheight,tiling,contrastmax,positionidx)
curr_time = datestr(now,'HH:MM:SS');
[tmp,curr_date] = weekday(sess_date);
%use chan order for generate channels string
chan_str = [];
cont_min = [];
cont_max = [];
ch_colors = [];
for i = 1:size(chan_ord,2)
    if i<size(chan_ord,2)
        chan_str = horzcat(chan_str,['    "',chan_ord{i},'",\n']);
        cont_min = horzcat(cont_min,'    0,\n');
        cont_max = horzcat(cont_max,['    ',contrastmax,',\n']);
        switch chan_ord{i}
            case '5-Violet-S'  %this is violet
                ch_colors = horzcat(ch_colors,'    -13421569,\n');
            case '6-Cyan-S'
                ch_colors = horzcat(ch_colors,'    -16724992,\n');
            case '7-Green-S'
                ch_colors = horzcat(ch_colors,'    -65485,\n');
            case '8-Red-S'
                ch_colors = horzcat(ch_colors,'    -1,\n');
            case '0-align-Q'  %alignment channel
                ch_colors = horzcat(ch_colors,'    -8,\n');
            otherwise
                warning('channel not recognized');
        end
    else   %last chan is different
        chan_str = horzcat(chan_str,['    "',chan_ord{i},'"\n']);
        cont_min = horzcat(cont_min,'    0\n');
        cont_max = horzcat(cont_max,['    ',contrastmax,'\n']);
        switch chan_ord{i}
            case '5-Violet-S'  %this is violet
                ch_colors = horzcat(ch_colors,'    -13421569\n');
            case '6-Cyan-S'
                ch_colors = horzcat(ch_colors,'    -16724992\n');
            case '7-Green-S'
                ch_colors = horzcat(ch_colors,'    -65485\n');
            case '8-Red-S'
                ch_colors = horzcat(ch_colors,'    -1\n');
            case '0-align-Q'  %alignment channel
                ch_colors = horzcat(ch_colors,'    -8\n');
            otherwise
                warning('channel not recognized');
        end
    end
end
metacap = ['"Summary": {\n',...
'  "Slices": ',num2str(sect),',\n',...
'  "UUID": "9c2e8f6f-8907-8689-b8a8-39d8f2ec8b83",\n',...
'  "UserName": "aratome",\n',...
'  "Depth": 2,\n',...
'  "PixelType": "GRAY16",\n'...
'  "Time": "',sess_date,' ',curr_date,' ',curr_time,' -0700",\n',...
'  "Date": "',sess_date,'",\n',...
'  "MetadataVersion": 10,\n',...
'  "PositionIndex": ',num2str(positionidx),',\n',...
'  "Width": ',imgwidth,',\n',...
'  "ChContrastMin": [\n',cont_min,...
'  ],\n',...
'  "StartTime": "',sess_date,' ',curr_date,' ',curr_time,' -0700",\n',...
'  "PixelAspect": 1,\n',...
'  "MicroManagerVersion": "1.4.10",\n',...
'  "ChNames": [\n',chan_str,...
'  ],\n',...
'  "IJType": 1,\n',...
'  "GridRow": 0,\n',...
'  "Comment": "",\n',...
'  "Height": ',imgheight,',\n',...
'  "NumComponents": 1,\n',...
'  "AT_DATA": "1",\n',...
'  "GridColumn": 0,\n',...
'  "Frames": 1,\n',...
'  "PixelSize_um": 0.1035,\n',...
'  "BitDepth": 16,\n',...
'  "ComputerName": "IMAGER",\n',...
'  "Channels": ',num2str(i),',\n',...
'  "Source": "Micro-Manager",\n',...
'  "ChColors": [\n',ch_colors,...
'  ],\n',...
'  "ChContrastMax": [\n',cont_max,...
'  ],\n',...
'  "Positions": ',num2str(tiling(1)*tiling(2)),'\n',...
'},'];
%--------------------------------------------------------------------------
%subfunction to generate metadata for each file in aratome format
function [metastr] = curr_file_meta(chnum,sect,ch,elptime,overlap,sess_date,imgwidth,curr_tile,ypos,imgheight,xpos,zpos,sectnum,positionidx)
curr_time = datestr(now,'HH:MM:SS');
[tmp,curr_date] = weekday(sess_date);
%parse the AT column and rows
tileidx = strfind(curr_tile,'-');
atcolumn = curr_tile(tileidx(2)+1:end);
atrow = curr_tile(tileidx(1)+1:tileidx(2)-1);
metastr = ['"FrameKey-0-',num2str(chnum-1),'-',num2str(sect),'": {\n',...
'  "ReImageCol": 0,\n',...
'  "Channel": "',ch,'",\n',...
'  "FrameIndex": 0,\n',...
'  "ElapsedTime-ms": ',num2str(elptime),',\n',...
'  "PixelType": "GRAY16",\n',...
'  "AT_Overlap_Pix": ',num2str(overlap),',\n',...
'  "Time": "',sess_date,' ',curr_date,' ',curr_time,' -0700",\n',...
'  "AT_Column": ',atcolumn,',\n',...
'  "FocusStatus": "AFC",\n',...
'  "Frame": 0,\n',...
'  "PositionIndex": ',num2str(positionidx),',\n',...
'  "Width": ',imgwidth,',\n',...
'  "AT_Z_Plane": 0,\n',...
'  "PositionName": "',curr_tile,'",\n',...
'  "YPositionUm": ',ypos,',\n',...
'  "Height": ',imgheight,',\n',...
'  "XPositionUm": ',xpos,',\n',...
'  "Z-Position": ',num2str(zpos),',\n',...
'  "AT_Z": ',num2str(888.888-zpos),',\n',...
'  "AT_Row": ',atrow,',\n',...
'  "Slice": ',num2str(sect),',\n',...
'  "ReImageSlice": 0,\n',...
'  "FileName": "img_000000000_',ch,'_',sectnum,'.tif",\n',...
'  "ReImageRow": 0,\n',...
'  "ChannelIndex": ',num2str(chnum),',\n',...
'  "SliceIndex": ',num2str(sect),',\n',...
'  "AT_Slice_Index": ',num2str(sect),',\n',...
'  "ZPositionUm": ',num2str(zpos),'\n',...
'},'];
%--------------------------------------------------------------------------
%subfunction to reorder the channel directories in a the proper aratome
%order
function [ch_ord,dir_ord] = ord_dir(curr_dir)
ch_ord = cell(size(curr_dir));
for i = 1:size(curr_dir,2)  %first go through and make a parallel channel group
    ch_ord{i} = ch_parse(curr_dir{i});
end
[ch_ord,ord_idx] = sort(ch_ord);  %sort it in order
%use the ord_idx to reorder the directory
dir_ord = curr_dir(ord_idx);
%--------------------------------------------------------------------------
%subfunction to parse the channels can convert channel to aratome format
function [curr_ch] = ch_parse(ch_str)
%couple of things: we are expecting the channel string to be in the formate
%of 'XXXX 350' we are going to use the last wavelength number as a switch
%parse string
wspace = strfind(ch_str,' ');
wavelength = ch_str(wspace(end)+1:end);
%pull characters as well, not using this, but here as an possible future
%feature
%characters = isstrprop(ch_str,'alpha');
%label = ch_str(characters);  %this will take only the letters.
switch wavelength
    case '350'  %this is violet
        curr_ch = '5-Violet-S';
    case '488'
        curr_ch = '6-Cyan-S';
    case {'594','590'}
        curr_ch = '7-Green-S';
    case '647'
        curr_ch = '8-Red-S';
    case '111'  %alignment channel
        curr_ch = '0-align-Q';
    otherwise
        warning('channel not recognized');
end
%--------------------------------------------------------------------------
%subfunction to generate metadata cap
function [metaroot] = root_meta(chan_ord)
%use chan order for generate channels string
chan_str = [];
for i = 1:size(chan_ord,2)
    if i<size(chan_ord,2)
        chan_str = horzcat(chan_str,['    {\n',...
'      "Name": "',chan_ord{i},'",\n',...
'      "HistogramMax": -1,\n',...
'      "Max": 4985,\n',...
'      "Gamma": 1,\n',...
'      "Min": 611,\n',...
'      "Color": -13421569\n',...
'    },\n']);
    else   %last chan is different
        chan_str = horzcat(chan_str,['    {\n',...
'      "Name": "',chan_ord{i},'",\n',...
'      "HistogramMax": -1,\n',...
'      "Max": 4985,\n',...
'      "Gamma": 1,\n',...
'      "Min": 611,\n',...
'      "Color": -13421569\n',...
'    }\n']);
    end
end
metaroot = ['{\n',...
'  "Channels": [\n',chan_str,...
'  ],\n',...
'  "Comments": {"Summary": ""}\n',...
'}'];
%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [overlap,sessiondate,dir_tmp,meta_dir,ksect,metaonly] = parse(input)

overlap = 10;    
sessiondate = datestr(now,'yyyy-mm-dd'); 
dir_tmp = [];
meta_dir = [];
ksect = 0; %by default don't maintain old section number
metaonly = 0;

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'sessiondate'
                    sessiondate = input{1,i+1};
                case 'overlap'
                    overlap = input{1,i+1};
%                     if ischar(input{1,i+1})
%                         overlap = input{1,i+1};
%                     else
%                         warning(['overlap must of char type']);
%                     end
                case 'dir'
                    dir_tmp = input{1,i+1};
                case 'meta'
                    meta_dir = input{1,i+1};
                case 'keepsection'
                    ksect = input{1,i+1};
                case 'metaonly'
                    metaonly = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end